<?php

declare(strict_types=1);

namespace App\Model;

use Nette;

class ServiceRepository
{
	public function __construct(
		private Nette\Database\Explorer $database,
	) {
	}

	public function findAll(): Nette\Database\Table\Selection
	{
		return $this->database->table('services')->order('weight ASC');
	}

	public function findActive(): Nette\Database\Table\Selection
	{
		return $this->findAll()->where('active', 1);
	}

	public function findByCategory(int $categoryId): Nette\Database\Table\Selection
	{
		return $this->findActive()->where('category_id', $categoryId);
	}

	public function findById(int $id): ?Nette\Database\Table\ActiveRow
	{
		return $this->database->table('services')->get($id) ?: null;
	}

	public function insert(array $values): Nette\Database\Table\ActiveRow
	{
		return $this->database->table('services')->insert($values);
	}

	public function update(int $id, array $values): void
	{
		$this->database->table('services')->where('id', $id)->update($values);
	}

	public function delete(int $id): void
	{
		$this->database->table('services')->where('id', $id)->delete();
	}

	/**
	 * Returns all active services grouped by category for the frontend.
	 */
	public function findGroupedByCategory(): array
	{
		$categories = $this->database->table('service_categories')
			->where('active', 1)
			->order('weight ASC')
			->fetchAll();

		$result = [];
		foreach ($categories as $cat) {
			$services = $this->database->table('services')
				->where('category_id', $cat->id)
				->where('active', 1)
				->order('weight ASC')
				->fetchAll();

			$result[] = [
				'category' => $cat,
				'services' => $services,
			];
		}

		return $result;
	}

	/**
	 * Returns flat list of active services for select box.
	 */
	public function findForSelect(): array
	{
		$categories = $this->database->table('service_categories')
			->where('active', 1)
			->order('weight ASC')
			->fetchAll();

		$result = ['' => '— Vyberte službu —'];
		foreach ($categories as $cat) {
			$services = $this->database->table('services')
				->where('category_id', $cat->id)
				->where('active', 1)
				->order('weight ASC')
				->fetchAll();

			foreach ($services as $svc) {
				$result[$svc->id] = $cat->icon . ' ' . $svc->name;
			}
		}

		return $result;
	}
}
